import { signInOrRegister, getAuth } from "./auth.js";
import { classifyBookmarks } from "./ai.js";

// ===== DOM Elements =====
const els = {
  grid: document.getElementById("grid"),
  emptyHint: document.getElementById("emptyHint"),
  searchForm: document.getElementById("searchForm"),
  searchInput: document.getElementById("searchInput"),
  wallpaper: document.getElementById("wallpaper"),
  sidebar: document.getElementById("sidebar"),
  sidebarToggle: document.getElementById("sidebarToggle"),
  sidebarClose: document.getElementById("sidebarClose"),
  sidebarTree: document.getElementById("sidebarTree"),
  sidebarFilter: document.getElementById("sidebarFilter"),
  engineBtn: document.getElementById("engineBtn"),
  engineIcon: document.getElementById("engineIcon"),
  engineDropdown: document.getElementById("engineDropdown"),
  btnAISort: document.getElementById("btnAISort"),
  btnAuth: document.getElementById("btnAuth"),
  btnSettings: document.getElementById("btnSettings"),
  btnTheme: document.getElementById("btnTheme"),
  prevPage: document.getElementById("prevPage"),
  nextPage: document.getElementById("nextPage"),
  pageIndicator: document.getElementById("pageIndicator"),
  // Settings dialog
  settingsDialog: document.getElementById("settingsDialog"),
  settingsClose: document.getElementById("settingsClose"),
  settingWallpaper: document.getElementById("settingWallpaper"),
  settingApiKey: document.getElementById("settingApiKey"),
  btnSaveSettings: document.getElementById("btnSaveSettings"),
  // Edit tile dialog
  editTileDialog: document.getElementById("editTileDialog"),
  editTileClose: document.getElementById("editTileClose"),
  editTileIcon: document.getElementById("editTileIcon"),
  editTileUrl: document.getElementById("editTileUrl"),
  editTileName: document.getElementById("editTileName"),
  editTileIconUrl: document.getElementById("editTileIconUrl"),
  editTileIconFile: document.getElementById("editTileIconFile"),
  editTileIndex: document.getElementById("editTileIndex"),
  editTileBgColor: document.getElementById("editTileBgColor"),
  colorPicker: document.getElementById("colorPicker"),
  btnSaveTile: document.getElementById("btnSaveTile"),
  btnResetTileIcon: document.getElementById("btnResetTileIcon"),
  iconSourceList: document.getElementById("iconSourceList"),
};

// Engine icons
const engineIcons = {
  google: "https://www.google.com/favicon.ico",
  bing: "https://www.bing.com/favicon.ico",
  baidu: "https://www.baidu.com/favicon.ico",
  duck: "https://duckduckgo.com/favicon.ico",
};

// ===== State =====
let allBookmarks = [];
let pinnedSites = [];
let currentEngine = "google";
let currentPage = 0;
let gridLayout = "8x5"; // Default layout
let itemsPerPage = 40; // Default 8*5

// Drag state
let draggedIndex = null;
let dragOverIndex = null;
let draggedElement = null;
let placeholderIndex = null;

// Edit dialog icon sources cache (for multi-source browser)
let currentEditIconSources = {};
let selectedIconSource = "letter"; // Currently selected source in icon browser (default to letter)

// ===== Init =====
init().catch(console.error);

async function init() {
  await loadSettings();
  await loadPinnedSites();
  const tree = await chrome.bookmarks.getTree();
  allBookmarks = flatten(tree);

  renderPinnedGrid();
  renderSidebarTree(tree);
  bindUI();
}

async function loadSettings() {
  const data = await chrome.storage.sync.get({
    wallpaperUrl: "",
    searchEngine: "google",
    theme: "dark",
    gridLayout: "8x5",
  });
  if (data.wallpaperUrl) {
    els.wallpaper.style.backgroundImage = `url("${data.wallpaperUrl}")`;
  }
  currentEngine = data.searchEngine;
  els.engineIcon.src = engineIcons[currentEngine] || engineIcons.google;

  // Apply saved theme
  applyTheme(data.theme);

  // Apply grid layout
  applyGridLayout(data.gridLayout);
}

function applyGridLayout(layout) {
  gridLayout = layout;
  const [cols, rows] = layout.split("x").map(Number);
  itemsPerPage = cols * rows;
  document.documentElement.style.setProperty("--grid-cols", cols);

  // Update settings UI
  const select = document.getElementById("settingGridLayout");
  if (select) select.value = layout;
}

// ===== Theme =====
function applyTheme(theme) {
  if (theme === "dark") {
    document.documentElement.classList.add("dark");
    els.btnTheme.title = "切换到亮色主题";
  } else {
    document.documentElement.classList.remove("dark");
    els.btnTheme.title = "切换到暗色主题";
  }
  updateThemeIcon();
}

async function toggleTheme() {
  const isDark = document.documentElement.classList.contains("dark");
  const newTheme = isDark ? "light" : "dark";
  applyTheme(newTheme);
  await chrome.storage.sync.set({ theme: newTheme });
}

// Default sites from screenshot
const defaultSites = [
  { title: "GitHub", url: "https://github.com" },
  { title: "Vercel", url: "https://vercel.com" },
  { title: "Cloudflare", url: "https://www.cloudflare.com" },
  { title: "React", url: "https://react.dev" },
  { title: "Tailwind CSS", url: "https://tailwindcss.com" },
  { title: "Next.js", url: "https://nextjs.org" },
  { title: "tRPC", url: "https://trpc.io" },
  { title: "Drizzle", url: "https://orm.drizzle.team" },
  { title: "Neon", url: "https://neon.tech" },
  { title: "Radix UI", url: "https://www.radix-ui.com" },
  { title: "shadcn/ui", url: "https://ui.shadcn.com" },
  { title: "Magic UI", url: "https://magicui.design" },
  { title: "Uiverse", url: "https://uiverse.io" },
  { title: "MDN", url: "https://developer.mozilla.org" },
  { title: "Three.js", url: "https://threejs.org" },
  { title: "Framer Motion", url: "https://www.framer.com/motion" },
  { title: "CodePen", url: "https://codepen.io" },
  { title: "TypeScript", url: "https://www.typescriptlang.org" },
  { title: "Twitter", url: "https://twitter.com" },
  { title: "YouTube", url: "https://www.youtube.com" },
  { title: "Reddit", url: "https://www.reddit.com" },
  { title: "Discord", url: "https://discord.com" },
  { title: "Facebook", url: "https://www.facebook.com" },
  { title: "Bilibili", url: "https://www.bilibili.com" },
  { title: "小红书", url: "https://www.xiaohongshu.com" },
  { title: "TikTok", url: "https://www.tiktok.com" },
  { title: "知乎", url: "https://www.zhihu.com" },
  { title: "Gmail", url: "https://mail.google.com" },
  { title: "QQ邮箱", url: "https://mail.qq.com" },
  { title: "微信公众平台", url: "https://mp.weixin.qq.com" },
  { title: "中交供应链", url: "https://sp.ccccltd.cn" },
  { title: "Google Maps", url: "https://maps.google.com" },
  { title: "Google AI", url: "https://ai.google" },
];

async function loadPinnedSites() {
  const { pinnedSites: stored } = await chrome.storage.sync.get({
    pinnedSites: null,
  });
  // If never set, use default sites
  if (stored === null) {
    pinnedSites = [...defaultSites];
    await savePinnedSites();
  } else {
    pinnedSites = stored;
  }
}

async function savePinnedSites() {
  await chrome.storage.sync.set({ pinnedSites });
}

// ===== Search =====
function buildSearchUrl(engine, q) {
  switch (engine) {
    case "bing":
      return `https://www.bing.com/search?q=${encodeURIComponent(q)}`;
    case "baidu":
      return `https://www.baidu.com/s?wd=${encodeURIComponent(q)}`;
    case "duck":
      return `https://duckduckgo.com/?q=${encodeURIComponent(q)}`;
    default:
      return `https://www.google.com/search?q=${encodeURIComponent(q)}`;
  }
}

// ===== Bookmarks Helper =====
function flatten(nodes, out = [], stack = []) {
  for (const n of nodes) {
    const path = n.title ? [...stack, n.title] : stack;
    if (n.url) out.push({ id: n.id, title: n.title, url: n.url, path });
    if (n.children) flatten(n.children, out, path);
  }
  return out;
}

// ===== Favicon (Unified priority, cache, normalize) =====
// Implements priority: user custom -> cache -> letter placeholder -> Clearbit -> HTML parse (background)
// Caches normalized icon (dataURL or URL) in chrome.storage.local with TTL

const ICON_CACHE_DB_NAME = "favicon_cache_db";
const ICON_CACHE_STORE_NAME = "icons";
const ICON_CACHE_TTL_MS = 7 * 24 * 60 * 60 * 1000; // 7 days

function faviconClearbit(url) {
  try {
    const u = new URL(url);
    return `https://logo.clearbit.com/${u.hostname}?size=256`;
  } catch {
    return "";
  }
}

async function fetchFaviconFromSite(url) {
  try {
    const response = await chrome.runtime.sendMessage({
      type: "FETCH_FAVICON",
      url: url,
    });
    if (response?.ok && response.iconUrl) return response.iconUrl;
    return null;
  } catch (err) {
    console.error("Failed to fetch favicon from site:", err);
    return null;
  }
}

function getDomainFromUrl(url) {
  try {
    return new URL(url).hostname;
  } catch {
    return url;
  }
}

// IndexedDB cache operations
function openIconCacheDB() {
  return new Promise((resolve, reject) => {
    const req = indexedDB.open(ICON_CACHE_DB_NAME, 1);
    req.onerror = () => reject(req.error);
    req.onsuccess = () => resolve(req.result);
    req.onupgradeneeded = (e) => {
      const db = e.target.result;
      if (!db.objectStoreNames.contains(ICON_CACHE_STORE_NAME)) {
        db.createObjectStore(ICON_CACHE_STORE_NAME);
      }
    };
  });
}

async function getIconCacheEntry(domain) {
  try {
    const db = await openIconCacheDB();
    return new Promise((resolve) => {
      const tx = db.transaction(ICON_CACHE_STORE_NAME, "readonly");
      const store = tx.objectStore(ICON_CACHE_STORE_NAME);
      const req = store.get(domain);
      req.onsuccess = () => {
        const entry = req.result;
        if (!entry) {
          resolve(null);
          return;
        }
        if (Date.now() - (entry.ts || 0) > ICON_CACHE_TTL_MS) {
          // Expired — clean up in background
          (async () => {
            try {
              const dbClean = await openIconCacheDB();
              const txClean = dbClean.transaction(
                ICON_CACHE_STORE_NAME,
                "readwrite"
              );
              txClean.objectStore(ICON_CACHE_STORE_NAME).delete(domain);
            } catch (e) {
              console.warn("Failed to clean expired icon cache", e);
            }
          })();
          resolve(null);
          return;
        }
        resolve(entry);
      };
      req.onerror = () => resolve(null);
    });
  } catch (err) {
    console.warn("Icon cache read failed:", err);
    return null;
  }
}

async function setIconCacheEntry(domain, entry) {
  try {
    // 严格限制缓存条目大小：只存储 URL 和元数据，不存储 dataURL（防止 quota 超限）
    const cacheEntry = {
      url: entry.url, // 远程 URL，浏览器会自动 HTTP 缓存
      source: entry.source, // "clearbit" | "html"
      ts: Date.now(),
    };
    const db = await openIconCacheDB();
    return new Promise((resolve, reject) => {
      const tx = db.transaction(ICON_CACHE_STORE_NAME, "readwrite");
      const store = tx.objectStore(ICON_CACHE_STORE_NAME);
      const req = store.put(cacheEntry, domain);
      req.onsuccess = () => resolve();
      req.onerror = () => reject(req.error);
    });
  } catch (err) {
    console.warn("Icon cache write failed:", err);
  }
}

async function testImageUrl(url) {
  if (!url) return false;
  try {
    // Try a lightweight fetch; mode no-cors may succeed but status is opaque.
    await fetch(url, { method: "HEAD", mode: "no-cors", cache: "no-store" });
    return true;
  } catch {
    return false;
  }
}

function applyIconToContainer(container, iconUrl, fallbackFn) {
  if (!container) return;
  container.innerHTML = "";
  const img = document.createElement("img");
  img.src = iconUrl;
  img.style.width = "100%";
  img.style.height = "100%";
  img.style.objectFit = "contain";
  if (fallbackFn) {
    img.onerror = fallbackFn;
  }
  container.appendChild(img);
}

// 注意：为避免 IndexedDB quota 超限，我们只在缓存中存储 URL，不存储 dataURL
// 浏览器会自动通过 HTTP 缓存 (Cache-Control/ETag) 缓存这些图标资源
// Core loader used across grid, sidebar and edit preview
async function loadAndApplyIcon(container, bm) {
  const { url, customIcon } = bm;
  // Priority: user custom -> cache -> Clearbit -> HTML parse -> keep letter
  const domain = getDomainFromUrl(url);

  const fallback = () => showLetterIcon(container, bm);

  // 1) User custom icon
  if (customIcon && customIcon !== "LETTER") {
    try {
      applyIconToContainer(container, customIcon, fallback);
      return;
    } catch {}
  }

  // 2) Check cache（仅存储 URL，不存储 dataURL）
  try {
    const cached = await getIconCacheEntry(domain);
    if (cached && cached.url) {
      applyIconToContainer(container, cached.url, fallback);
      return;
    }
  } catch (err) {
    console.warn("Icon cache read failed:", err);
  }

  // Keep letter placeholder shown by caller; then try async replacements
  (async () => {
    // 3) Try Clearbit (best effort)
    try {
      const cb = faviconClearbit(url);
      if (cb && (await testImageUrl(cb))) {
        // 存储 URL，不存储 dataURL（避免 quota 超限）
        await setIconCacheEntry(domain, { url: cb, source: "clearbit" });
        applyIconToContainer(container, cb, fallback);
        return;
      }
    } catch (err) {
      console.warn("Clearbit attempt failed", err);
    }

    // 4) Try HTML parsing via background fetch
    try {
      const htmlIcon = await fetchFaviconFromSite(url);
      if (htmlIcon && (await testImageUrl(htmlIcon))) {
        // 存储 URL，不存储 dataURL
        await setIconCacheEntry(domain, { url: htmlIcon, source: "html" });
        applyIconToContainer(container, htmlIcon, fallback);
        return;
      }
    } catch (err) {
      console.warn("HTML parse attempt failed", err);
    }

    // 5) Try DuckDuckGo
    try {
      const ddgUrl = `https://icons.duckduckgo.com/ip3/${
        new URL(url).hostname
      }.ico`;
      if (await testImageUrl(ddgUrl)) {
        await setIconCacheEntry(domain, { url: ddgUrl, source: "duckduckgo" });
        applyIconToContainer(container, ddgUrl, fallback);
        return;
      }
    } catch (err) {
      console.warn("DuckDuckGo attempt failed", err);
    }
  })();
}

function updateThemeIcon() {
  const isDark = document.documentElement.classList.contains("dark");
  const moonIcon = document.querySelector(".theme-icon-moon");
  const sunIcon = document.querySelector(".theme-icon-sun");

  if (moonIcon && sunIcon) {
    if (isDark) {
      moonIcon.style.display = "none";
      sunIcon.style.display = "block";
    } else {
      moonIcon.style.display = "block";
      sunIcon.style.display = "none";
    }
  }
}

// Backwards-compatible wrappers used by UI
async function loadFaviconWithFallback(iconContainer, bm) {
  return loadAndApplyIcon(iconContainer, bm);
}

async function loadSidebarFavicon(iconWrap, node) {
  return loadAndApplyIcon(iconWrap, node);
}

async function loadPreviewFavicon(bm, bgColor) {
  // The preview element is els.editTileIcon
  const container = els.editTileIcon;
  // If user provided explicit icon URL in the edit input, prefer that
  const userUrl = els.editTileIconUrl.value || bm.customIcon;
  if (userUrl && userUrl !== "LETTER") {
    // Try to apply user URL immediately
    try {
      container.innerHTML = "";
      const img = document.createElement("img");
      img.src = userUrl;
      img.style.width = "100%";
      img.style.height = "100%";
      img.style.objectFit = "contain";
      img.onerror = () => showPreviewLetter(bm, bgColor);
      container.appendChild(img);
      return;
    } catch (err) {
      console.warn("Preview user icon failed", err);
    }
  }

  // Otherwise show letter (already done by caller) and try async replacement
  return loadAndApplyIcon(container, { url: bm.url });
}

// Get first two characters from title for letter icon
function getTwoChars(text) {
  if (!text) return "??";
  // Remove common prefixes and clean up
  const cleaned = text.trim();
  // Match first two characters (letters, numbers, or emoji)
  const matches = cleaned.match(/[\p{L}\p{N}\p{Emoji}]/gu);
  if (matches && matches.length >= 2) {
    return matches.slice(0, 2).join("");
  } else if (matches && matches.length === 1) {
    return matches[0];
  }
  return cleaned.slice(0, 2) || "??";
}

// Predefined theme colors for letter icons
const ICON_COLORS = [
  "#6366f1", // Indigo
  "#8b5cf6", // Violet
  "#ec4899", // Pink
  "#f43f5e", // Rose
  "#ef4444", // Red
  "#f97316", // Orange
  "#eab308", // Yellow
  "#84cc16", // Lime
  "#22c55e", // Green
  "#14b8a6", // Teal
  "#06b6d4", // Cyan
  "#3b82f6", // Blue
  "#6b7280", // Gray
];

// Get a consistent color based on URL (so same site always gets same color)
function getIconColor(url) {
  let hash = 0;
  for (let i = 0; i < url.length; i++) {
    hash = url.charCodeAt(i) + ((hash << 5) - hash);
  }
  return ICON_COLORS[Math.abs(hash) % ICON_COLORS.length];
}

// Alias for getIconColor (used in sidebar)
const hashColor = getIconColor;

// ===== Render Pinned Grid with Pagination =====
function getTotalPages() {
  return Math.ceil((pinnedSites.length + 1) / itemsPerPage) || 1;
}

function renderPinnedGrid() {
  els.emptyHint.classList.add("hidden");

  // Calculate page bounds
  const totalPages = getTotalPages();
  if (currentPage >= totalPages) currentPage = totalPages - 1;
  if (currentPage < 0) currentPage = 0;

  const start = currentPage * itemsPerPage;
  const end = start + itemsPerPage;

  const tiles = [];
  for (let i = start; i < end; i++) {
    if (i < pinnedSites.length) {
      tiles.push(createTile(pinnedSites[i], i));
    } else if (i === pinnedSites.length) {
      tiles.push(createAddTile());
    }
  }

  els.grid.replaceChildren(...tiles);

  updatePaginationUI();
}

function createAddTile() {
  const div = document.createElement("div");
  div.className = "tile add-tile";
  div.innerHTML = `
    <div class="tile-icon" style="background: var(--card); color: var(--foreground);">
      <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <line x1="12" y1="5" x2="12" y2="19"></line>
        <line x1="5" y1="12" x2="19" y2="12"></line>
      </svg>
    </div>
    <div class="tile-title">添加网站</div>
  `;
  div.addEventListener("click", () => {
    openEditTileDialog(-1);
  });
  return div;
}

function updatePaginationUI() {
  const totalPages = getTotalPages();

  // Update arrow buttons
  els.prevPage.disabled = currentPage === 0;
  els.nextPage.disabled =
    currentPage >= totalPages - 1 || pinnedSites.length === 0;

  // Update page indicator dots
  els.pageIndicator.innerHTML = "";
  if (totalPages > 1) {
    for (let i = 0; i < totalPages; i++) {
      const dot = document.createElement("div");
      dot.className = "page-dot" + (i === currentPage ? " active" : "");
      dot.addEventListener("click", () => {
        currentPage = i;
        renderPinnedGrid();
      });
      els.pageIndicator.appendChild(dot);
    }
  }
}

// Update visual shift classes during drag
function updateDragVisuals() {
  const tiles = Array.from(els.grid.querySelectorAll(".tile"));

  tiles.forEach((tile) => {
    const tileIdx = parseInt(tile.dataset.index);
    tile.classList.remove("shift-left", "shift-right");

    if (draggedIndex === null || placeholderIndex === null) return;
    if (tileIdx === draggedIndex) return;

    // Determine if this tile should shift
    if (draggedIndex < placeholderIndex) {
      // Dragging right: tiles between original and placeholder shift left
      if (tileIdx > draggedIndex && tileIdx <= placeholderIndex) {
        tile.classList.add("shift-left");
      }
    } else if (draggedIndex > placeholderIndex) {
      // Dragging left: tiles between placeholder and original shift right
      if (tileIdx >= placeholderIndex && tileIdx < draggedIndex) {
        tile.classList.add("shift-right");
      }
    }
  });
}

function createTile(bm, idx) {
  const a = document.createElement("a");
  a.className = "tile";
  a.href = bm.url;
  a.target = "_blank";
  a.rel = "noreferrer noopener";
  a.draggable = true;
  a.dataset.index = idx;

  // Drag events
  a.addEventListener("dragstart", (e) => {
    e.stopPropagation();
    draggedIndex = idx;
    draggedElement = a;
    placeholderIndex = idx;

    // Delay adding the dragging class so the browser captures the element as the drag image first
    setTimeout(() => a.classList.add("dragging"), 0);

    e.dataTransfer.effectAllowed = "move";
    e.dataTransfer.setData("text/plain", idx.toString());

    // Removed: Create empty drag image to prevent default ghost image
    // const emptyImage = new Image();
    // e.dataTransfer.setDragImage(emptyImage, 0, 0);
  });

  a.addEventListener("dragend", async () => {
    a.classList.remove("dragging");
    document.querySelectorAll(".tile").forEach((t) => {
      t.classList.remove("drag-over", "shift-left", "shift-right");
    });

    // Apply the reorder if placeholder moved
    if (placeholderIndex !== null && draggedIndex !== placeholderIndex) {
      const draggedItem = pinnedSites[draggedIndex];
      pinnedSites.splice(draggedIndex, 1);
      pinnedSites.splice(placeholderIndex, 0, draggedItem);
      await savePinnedSites();
    }

    draggedIndex = null;
    dragOverIndex = null;
    draggedElement = null;
    placeholderIndex = null;
    renderPinnedGrid();
  });

  a.addEventListener("dragover", (e) => {
    e.preventDefault();
    if (draggedIndex === null || draggedIndex === idx) return;

    const rect = a.getBoundingClientRect();
    const midX = rect.left + rect.width / 2;

    // Determine new placeholder index based on mouse position and current dragged position
    let newPlaceholderIndex = idx;
    if (draggedIndex < idx) {
      // Dragging right - insert before if mouse is before midpoint
      if (e.clientX < midX && idx > draggedIndex) {
        newPlaceholderIndex = idx - 1;
      }
    } else if (draggedIndex > idx) {
      // Dragging left - insert after if mouse is after midpoint
      if (e.clientX > midX && idx < draggedIndex) {
        newPlaceholderIndex = idx + 1;
      }
    }

    if (newPlaceholderIndex !== placeholderIndex) {
      placeholderIndex = newPlaceholderIndex;
      updateDragVisuals();
    }
  });

  a.addEventListener("dragleave", () => {
    a.classList.remove("drag-over");
  });

  a.addEventListener("drop", (e) => {
    e.preventDefault();
  });

  const icon = document.createElement("div");
  icon.className = "tile-icon";

  // 图标显示逻辑：
  // customIcon = "LETTER" → 文字图标
  // customIcon = URL → 自定义图标
  // customIcon = undefined → 先显示文字，然后异步加载缓存图标

  if (bm.customIcon === "LETTER") {
    showLetterIcon(icon, bm);
  } else if (bm.customIcon) {
    const img = document.createElement("img");
    img.src = bm.customIcon;
    img.onerror = () => showLetterIcon(icon, bm);
    icon.appendChild(img);
  } else {
    // 没有customIcon，先显示文字占位
    showLetterIcon(icon, bm);
    // 然后尝试从缓存加载
    loadFaviconWithFallback(icon, bm).catch(() => {});
  }

  const name = document.createElement("div");
  name.className = "tile-name";
  name.textContent = bm.title || new URL(bm.url).hostname;

  // Edit button
  const editBtn = document.createElement("button");
  editBtn.className = "tile-edit";
  editBtn.innerHTML =
    '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path></svg>';
  editBtn.title = "编辑";
  editBtn.addEventListener("click", (e) => {
    e.preventDefault();
    e.stopPropagation();
    openEditTileDialog(idx);
  });

  const removeBtn = document.createElement("button");
  removeBtn.className = "tile-remove";
  removeBtn.innerHTML =
    '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>';
  removeBtn.title = "移除";
  removeBtn.addEventListener("click", async (e) => {
    e.preventDefault();
    e.stopPropagation();
    pinnedSites.splice(idx, 1);
    await savePinnedSites();
    renderPinnedGrid();
    updateSidebarAddButtons();
  });

  a.append(icon, name, editBtn, removeBtn);
  return a;
}

// Show letter icon fallback (uses first 2 characters of title)
function showLetterIcon(container, bm) {
  container.innerHTML = "";
  const letter = document.createElement("div");
  letter.className = "tile-letter";
  letter.textContent = getTwoChars(bm.title || new URL(bm.url).hostname);
  // Use custom background color if set, otherwise use consistent color based on URL
  letter.style.background = bm.iconBgColor || getIconColor(bm.url);
  container.appendChild(letter);
}

// Open edit tile dialog
function openEditTileDialog(idx) {
  els.editTileIndex.value = idx;

  if (idx === -1) {
    // New tile
    els.editTileName.value = "";
    els.editTileUrl.value = "";
    els.editTileIconUrl.value = "";
    els.editTileBgColor.value = "";
    selectedIconSource = "letter";

    // Reset preview
    els.editTileIcon.innerHTML = "";
    const letter = document.createElement("div");
    letter.className = "tile-letter";
    letter.textContent = "+";
    letter.style.background = "var(--primary)";
    els.editTileIcon.appendChild(letter);

    updateColorPickerSelection("");

    // Clear icon source list
    document.getElementById("iconSourceList").innerHTML = "";
  } else {
    const bm = pinnedSites[idx];
    els.editTileName.value = bm.title || "";
    els.editTileUrl.value = bm.url || "";
    els.editTileIconUrl.value = bm.customIcon || "";
    els.editTileBgColor.value = bm.iconBgColor || "";

    // Set initial selected source based on current icon
    if (bm.customIcon === "LETTER") {
      selectedIconSource = "letter";
    } else if (bm.customIcon) {
      // 检查是否来自某个已知来源
      selectedIconSource = "upload"; // 默认认为是上传的
    } else {
      // 没有customIcon，检查缓存中的来源
      const domain = getDomainFromUrl(bm.url);
      getIconCacheEntry(domain)
        .then((cached) => {
          if (cached?.source) {
            selectedIconSource = cached.source; // clearbit/html/duckduckgo
          } else {
            selectedIconSource = "letter"; // 默认
          }
          // 重新渲染以更新选中状态
          renderIconSourceBrowser(bm);
        })
        .catch(() => {
          selectedIconSource = "letter";
        });
      selectedIconSource = "letter"; // 临时设置，等待异步结果
    }

    // Update color picker selection
    updateColorPickerSelection(bm.iconBgColor || getIconColor(bm.url));

    // Update preview
    updateEditTilePreview(bm);

    // Fetch all available icon sources in parallel
    fetchAllIconSources(bm);
  }

  els.editTileDialog.showModal();
}

// Update color picker UI to show selected color
function updateColorPickerSelection(selectedColor) {
  document.querySelectorAll(".color-option").forEach((opt) => {
    opt.classList.toggle("selected", opt.dataset.color === selectedColor);
  });
}

function updateEditTilePreview(bm) {
  const iconUrl = els.editTileIconUrl.value || bm.customIcon;
  const bgColor =
    els.editTileBgColor.value || bm.iconBgColor || getIconColor(bm.url);

  // If user chose letter icon or iconUrl is LETTER marker
  if (iconUrl === "LETTER" || selectedIconSource === "letter") {
    showPreviewLetter(bm, bgColor);
    return;
  }

  if (iconUrl) {
    const img = document.createElement("img");
    img.src = iconUrl;
    img.style.cssText = "width: 100%; height: 100%; object-fit: contain;";

    // Immediate update attempt
    els.editTileIcon.innerHTML = "";
    els.editTileIcon.appendChild(img);

    img.onerror = () => {
      showPreviewLetter(bm, bgColor);
    };
  } else {
    // No custom icon, show letter based on current selection
    showPreviewLetter(bm, bgColor);
  }
}

// Show letter preview in edit dialog
function showPreviewLetter(bm, bgColor) {
  els.editTileIcon.innerHTML = "";
  const letter = document.createElement("div");
  letter.className = "tile-letter";
  const title = els.editTileName.value || bm.title || new URL(bm.url).hostname;
  letter.textContent = getTwoChars(title);
  letter.style.background = bgColor;
  els.editTileIcon.appendChild(letter);
}

// Fetch all available icon sources (5种完整来源)
async function fetchAllIconSources(bm) {
  const hasUploadedIcon = bm.customIcon && bm.customIcon !== "LETTER";
  const domain = getDomainFromUrl(bm.url);

  // 从缓存读取已有的图标
  let cachedEntry = null;
  try {
    cachedEntry = await getIconCacheEntry(domain);
  } catch (err) {
    console.warn("Failed to read cache:", err);
  }

  currentEditIconSources = {
    letter: getTwoChars(bm.title || new URL(bm.url).hostname),
    duckduckgo: cachedEntry?.source === "duckduckgo" ? cachedEntry.url : null,
    html: cachedEntry?.source === "html" ? cachedEntry.url : null,
    clearbit: cachedEntry?.source === "clearbit" ? cachedEntry.url : null,
    upload: hasUploadedIcon ? bm.customIcon : null,
    // Loading states
    duckduckgoLoading: false,
    duckduckgoError: false,
    htmlLoading: false,
    htmlError: false,
    clearbitLoading: false,
    clearbitError: false,
  };

  renderIconSourceBrowser(bm);

  // 自动获取所有未缓存的图标
  await Promise.all([
    fetchIconIfNeeded("duckduckgo", bm),
    fetchIconIfNeeded("html", bm),
    fetchIconIfNeeded("clearbit", bm),
  ]);
}

// 按需获取图标
async function fetchIconIfNeeded(sourceKey, bm) {
  const sources = currentEditIconSources;
  if (sources[sourceKey]) return; // 已有缓存

  sources[`${sourceKey}Loading`] = true;
  renderIconSourceBrowser(bm);

  try {
    let iconUrl = null;
    const domain = getDomainFromUrl(bm.url);
    let success = false;

    if (sourceKey === "clearbit") {
      iconUrl = faviconClearbit(bm.url);
      if (iconUrl && (await testImageUrl(iconUrl))) {
        await setIconCacheEntry(domain, { url: iconUrl, source: "clearbit" });
        sources.clearbit = iconUrl;
        success = true;
      } else {
        sources.clearbitError = true;
      }
    } else if (sourceKey === "html") {
      iconUrl = await fetchFaviconFromSite(bm.url);
      if (iconUrl && (await testImageUrl(iconUrl))) {
        await setIconCacheEntry(domain, { url: iconUrl, source: "html" });
        sources.html = iconUrl;
        success = true;
      } else {
        sources.htmlError = true;
      }
    } else if (sourceKey === "duckduckgo") {
      iconUrl = `https://icons.duckduckgo.com/ip3/${
        new URL(bm.url).hostname
      }.ico`;
      if (await testImageUrl(iconUrl)) {
        await setIconCacheEntry(domain, { url: iconUrl, source: "duckduckgo" });
        sources.duckduckgo = iconUrl;
        success = true;
      } else {
        sources.duckduckgoError = true;
      }
    }

    // 如果当前选中的正是这个失败的源，回退到文字图标
    if (!success && selectedIconSource === sourceKey) {
      selectIconSource("letter", bm);
    }
  } catch (err) {
    console.warn(`${sourceKey} fetch failed:`, err);
    sources[`${sourceKey}Error`] = true;
    if (selectedIconSource === sourceKey) {
      selectIconSource("letter", bm);
    }
  }

  sources[`${sourceKey}Loading`] = false;
  renderIconSourceBrowser(bm);
} // Render icon source browser (shows available sources as selectable cards)
function renderIconSourceBrowser(bm) {
  const sources = currentEditIconSources;
  els.iconSourceList.innerHTML = "";

  // 5种图标来源（按优先级排序）
  const sourceDefinitions = [
    { key: "letter", title: "文字" },
    { key: "duckduckgo", title: "DuckDuckGo" },
    { key: "html", title: "网站解析" },
    { key: "clearbit", title: "Clearbit" },
    // { key: "upload", title: "自定义" } // Removed as requested
  ];

  // Render all sources
  sourceDefinitions.forEach((def) => {
    const card = document.createElement("div");
    card.className = "icon-source-card";
    card.dataset.source = def.key;

    // 检查是否选中
    if (def.key === selectedIconSource) {
      card.classList.add("selected");
    }

    // 检查是否失败/禁用
    const hasError = sources[`${def.key}Error`];
    const isDisabled = hasError || (def.key === "upload" && !sources.upload);
    if (isDisabled && def.key !== "letter") {
      card.classList.add("disabled");
    }

    const iconDiv = document.createElement("div");
    iconDiv.className = "icon-source-icon";

    // Display the icon preview based on type
    if (def.key === "letter") {
      const letterDiv = document.createElement("div");
      letterDiv.style.cssText = `
        width: 100%;
        height: 100%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 24px;
        font-weight: bold;
        background: ${bm.iconBgColor || getIconColor(bm.url)};
        color: white;
        border-radius: 4px;
      `;
      letterDiv.textContent = sources.letter;
      iconDiv.appendChild(letterDiv);
    } else if (def.key === "upload") {
      if (sources.upload) {
        const img = document.createElement("img");
        img.src = sources.upload;
        img.style.cssText = "width: 100%; height: 100%; object-fit: contain;";
        iconDiv.appendChild(img);
      } else {
        iconDiv.innerHTML =
          '<div style="font-size:24px;color:var(--muted-foreground);">➕</div>';
      }
    } else {
      // clearbit / html / duckduckgo
      const isLoading = sources[`${def.key}Loading`];
      const iconUrl = sources[def.key];

      if (isLoading) {
        iconDiv.innerHTML =
          '<div style="font-size:20px;color:var(--muted-foreground);">⏳</div>';
      } else if (hasError) {
        iconDiv.innerHTML =
          '<div style="font-size:20px;color:#999;opacity:0.5;">✕</div>';
      } else if (iconUrl) {
        const img = document.createElement("img");
        img.src = iconUrl;
        img.style.cssText = "width: 100%; height: 100%; object-fit: contain;";
        img.onerror = () => {
          iconDiv.innerHTML =
            '<div style="font-size:20px;color:#999;opacity:0.5;">✕</div>';
          sources[`${def.key}Error`] = true;
          card.classList.add("disabled");
          if (selectedIconSource === def.key) {
            selectIconSource("letter", bm);
          }
        };
        iconDiv.appendChild(img);
      } else {
        iconDiv.innerHTML =
          '<div style="font-size:20px;color:var(--muted-foreground);">⏳</div>';
      }
    }

    card.appendChild(iconDiv);

    // Click handler
    card.addEventListener("click", async () => {
      // 禁止点击失败/禁用的卡片
      if (card.classList.contains("disabled")) return;

      if (def.key === "upload") {
        els.editTileIconFile.click();
      } else if (def.key === "letter") {
        selectIconSource("letter", bm);
      } else if (sources[def.key]) {
        // 已有图标，直接选择
        selectIconSource(def.key, bm);
      }
    });

    els.iconSourceList.appendChild(card);
  });
}

// Select an icon source and apply it
async function selectIconSource(source, bm) {
  selectedIconSource = source;
  const sourceValue = currentEditIconSources[source];

  // Update the form based on selection
  if (source === "letter") {
    els.editTileIconUrl.value = "LETTER";
  } else if (
    source === "upload" ||
    source === "clearbit" ||
    source === "html" ||
    source === "duckduckgo"
  ) {
    // 存储具体的URL，并标记来源
    els.editTileIconUrl.value = sourceValue || "";
  }

  // Update preview
  updateEditTilePreview(bm);

  // Update visual selection in browser
  document.querySelectorAll(".icon-source-card").forEach((card) => {
    card.classList.toggle("selected", card.dataset.source === source);
  });

  // Live update the grid tile immediately
  const idx = parseInt(els.editTileIndex.value);
  if (!isNaN(idx) && pinnedSites[idx]) {
    // Temporarily update the object in memory
    pinnedSites[idx].customIcon = els.editTileIconUrl.value || undefined;
    // We don't save to storage yet (that happens on close), but we re-render the grid
    renderPinnedGrid();
  }
}

// ===== Render Sidebar Tree =====
function renderSidebarTree(tree, filter = "") {
  const root = tree[0];
  const container = document.createElement("div");
  if (root.children) {
    for (const child of root.children) {
      renderFolderOrItem(child, container, filter.toLowerCase());
    }
  }
  els.sidebarTree.replaceChildren(container);
}

function renderFolderOrItem(node, parent, filter) {
  if (node.children) {
    const folder = document.createElement("div");
    folder.className = "sb-folder";

    const header = document.createElement("div");
    header.className = "sb-folder-header";
    header.innerHTML = `<span class="arrow"><svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"></polyline></svg></span> <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="margin-right:4px"><path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"></path></svg> ${
      node.title || "未命名"
    }`;
    header.addEventListener("click", () =>
      folder.classList.toggle("collapsed")
    );

    const children = document.createElement("div");
    children.className = "sb-folder-children";

    let hasVisibleChild = false;
    for (const child of node.children) {
      if (renderFolderOrItem(child, children, filter)) hasVisibleChild = true;
    }

    if (!filter || hasVisibleChild) {
      folder.append(header, children);
      parent.appendChild(folder);
      return true;
    }
    return false;
  } else if (node.url) {
    const title = (node.title || "").toLowerCase();
    const url = (node.url || "").toLowerCase();
    if (filter && !title.includes(filter) && !url.includes(filter))
      return false;

    const item = document.createElement("div");
    item.className = "sb-item";
    item.draggable = true;
    item.dataset.url = node.url;
    item.dataset.title = node.title;

    const iconWrap = document.createElement("div");
    iconWrap.className = "sb-item-icon";

    // Start with letter icon, then try to load real favicon in background
    iconWrap.textContent = getTwoChars(node.title);
    iconWrap.style.background = hashColor(node.url);
    iconWrap.style.color = "#fff";
    iconWrap.style.fontSize = "10px";
    iconWrap.style.display = "grid";
    iconWrap.style.placeItems = "center";

    // Try to load favicon in background (silently)
    loadSidebarFavicon(iconWrap, node);

    const nameSpan = document.createElement("span");
    nameSpan.className = "sb-item-name";
    nameSpan.textContent = node.title || new URL(node.url).hostname;
    nameSpan.title = node.url;

    const addBtn = document.createElement("button");
    addBtn.className = "sb-item-add";
    addBtn.dataset.url = node.url;
    const isPinned = pinnedSites.some((p) => p.url === node.url);
    addBtn.innerHTML = isPinned
      ? '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg>'
      : '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><line x1="12" y1="5" x2="12" y2="19"></line><line x1="5" y1="12" x2="19" y2="12"></line></svg>';
    if (isPinned) addBtn.classList.add("added");
    addBtn.title = isPinned ? "已添加" : "添加到首页";
    addBtn.addEventListener("click", async (e) => {
      e.stopPropagation();
      if (isPinned) return;
      pinnedSites.push({ id: node.id, title: node.title, url: node.url });
      await savePinnedSites();
      renderPinnedGrid();
      addBtn.innerHTML =
        '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg>';
      addBtn.classList.add("added");
    });

    const deleteBtn = document.createElement("button");
    deleteBtn.className = "sb-item-delete";
    deleteBtn.title = "删除书签";
    deleteBtn.innerHTML =
      '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg>';
    deleteBtn.addEventListener("click", async (e) => {
      e.stopPropagation();
      if (confirm(`确定要删除书签 "${node.title}" 吗？`)) {
        try {
          await chrome.bookmarks.remove(node.id);
          item.remove();
          // Also remove from pinned if exists
          const pinnedIdx = pinnedSites.findIndex((p) => p.url === node.url);
          if (pinnedIdx !== -1) {
            pinnedSites.splice(pinnedIdx, 1);
            await savePinnedSites();
            renderPinnedGrid();
          }
        } catch (err) {
          console.error("Failed to delete bookmark:", err);
          alert("删除失败，请重试");
        }
      }
    });

    item.addEventListener("click", () => window.open(node.url, "_blank"));

    // Drag events for sidebar items - allow dragging to grid
    item.addEventListener("dragstart", (e) => {
      e.stopPropagation();
      e.dataTransfer.effectAllowed = "move";
      e.dataTransfer.setData("text/x-url", node.url);
      e.dataTransfer.setData("text/x-title", node.title);
      item.style.opacity = "0.5";
    });

    item.addEventListener("dragend", () => {
      item.style.opacity = "1";
    });

    item.append(iconWrap, nameSpan, addBtn, deleteBtn);
    parent.appendChild(item);
    return true;
  }
  return false;
}

function updateSidebarAddButtons() {
  document.querySelectorAll(".sb-item-add").forEach((btn) => {
    const url = btn.dataset.url;
    const isPinned = pinnedSites.some((p) => p.url === url);
    btn.innerHTML = isPinned
      ? '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg>'
      : '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><line x1="12" y1="5" x2="12" y2="19"></line><line x1="5" y1="12" x2="19" y2="12"></line></svg>';
    btn.classList.toggle("added", isPinned);
  });
}

// ===== UI Bindings =====
async function saveAndCloseEditDialog() {
  // Save changes before closing
  const idx = parseInt(els.editTileIndex.value);
  const newName = els.editTileName.value.trim();
  const newUrl = els.editTileUrl.value.trim();
  const customIcon = els.editTileIconUrl.value.trim();
  const bgColor = els.editTileBgColor.value.trim();

  if (idx === -1) {
    // Add new site
    if (newUrl) {
      // Ensure URL has protocol
      let finalUrl = newUrl;
      if (!/^https?:\/\//i.test(finalUrl)) {
        finalUrl = "https://" + finalUrl;
      }

      const title = newName || getDomainFromUrl(finalUrl);
      const newSite = {
        title: title,
        url: finalUrl,
        customIcon: customIcon || undefined,
        iconBgColor: bgColor || undefined,
      };
      pinnedSites.push(newSite);

      // Add to bookmarks
      try {
        // Try to add to Bookmarks Bar (ID "1")
        await chrome.bookmarks.create({
          parentId: "1",
          title: title,
          url: finalUrl,
        });
      } catch (e) {
        console.error("Failed to add bookmark to bar, trying default", e);
        try {
          await chrome.bookmarks.create({
            title: title,
            url: finalUrl,
          });
        } catch (e2) {
          console.error("Failed to add bookmark", e2);
        }
      }
    }
  } else {
    // Update existing site
    if (newName) {
      pinnedSites[idx].title = newName;
    }
    if (newUrl) {
      pinnedSites[idx].url = newUrl;
    }
    pinnedSites[idx].customIcon = customIcon || undefined;
    pinnedSites[idx].iconBgColor = bgColor || undefined;
  }

  await savePinnedSites();
  renderPinnedGrid();
  els.editTileDialog.close();
}

function bindUI() {
  // Grid drag events - allow dropping on the grid
  els.grid.addEventListener("dragover", (e) => {
    e.preventDefault();
    e.dataTransfer.dropEffect = "move";
  });

  els.grid.addEventListener("drop", async (e) => {
    e.preventDefault();

    // Check if dragging from sidebar
    const url = e.dataTransfer.getData("text/x-url");
    const title = e.dataTransfer.getData("text/x-title");

    if (url && title) {
      // Adding from sidebar - find insertion position based on mouse coordinates
      const exists = pinnedSites.some((p) => p.url === url);
      if (!exists) {
        // Get the grid position from mouse coordinates
        const gridRect = els.grid.getBoundingClientRect();
        const relX = e.clientX - gridRect.left;
        const relY = e.clientY - gridRect.top;

        // Find which tile the cursor is over
        let insertIndex = pinnedSites.length; // default to end
        const tiles = Array.from(els.grid.querySelectorAll(".tile"));
        for (const tile of tiles) {
          const tileRect = tile.getBoundingClientRect();
          const tileRelX = tileRect.left - gridRect.left;
          const tileRelY = tileRect.top - gridRect.top;

          // Check if mouse is over this tile
          if (
            relX >= tileRelX &&
            relX < tileRelX + tileRect.width &&
            relY >= tileRelY &&
            relY < tileRelY + tileRect.height
          ) {
            const tileIndex = parseInt(tile.dataset.index);
            insertIndex = tileIndex;
            break;
          }
        }

        pinnedSites.splice(insertIndex, 0, {
          url: url,
          title: title,
        });
        await savePinnedSites();
        renderPinnedGrid();
        updateSidebarAddButtons();
      }
    }
  });

  // Sidebar
  els.sidebarToggle.addEventListener("click", (e) => {
    e.stopPropagation();
    els.sidebar.classList.add("open");
  });
  els.sidebarClose.addEventListener("click", () =>
    els.sidebar.classList.remove("open")
  );

  // Click outside sidebar to close
  document.addEventListener("click", (e) => {
    if (
      els.sidebar.classList.contains("open") &&
      !els.sidebar.contains(e.target) &&
      !els.sidebarToggle.contains(e.target)
    ) {
      els.sidebar.classList.remove("open");
    }
  });

  els.sidebarFilter.addEventListener("input", (e) => {
    chrome.bookmarks
      .getTree()
      .then((tree) => renderSidebarTree(tree, e.target.value));
  });

  // Pagination
  els.prevPage.addEventListener("click", () => {
    if (currentPage > 0) {
      currentPage--;
      renderPinnedGrid();
    }
  });

  els.nextPage.addEventListener("click", () => {
    if (currentPage < getTotalPages() - 1) {
      currentPage++;
      renderPinnedGrid();
    }
  });

  // Keyboard navigation for pages
  document.addEventListener("keydown", (e) => {
    if (e.target.tagName === "INPUT") return;
    if (e.key === "ArrowLeft" && currentPage > 0) {
      currentPage--;
      renderPinnedGrid();
    } else if (e.key === "ArrowRight" && currentPage < getTotalPages() - 1) {
      currentPage++;
      renderPinnedGrid();
    }
  });

  // Search
  els.searchForm.addEventListener("submit", (e) => {
    e.preventDefault();
    const q = els.searchInput.value.trim();
    if (!q) return;
    window.location.href = buildSearchUrl(currentEngine, q);
  });

  // Engine dropdown
  els.engineBtn.addEventListener("click", (e) => {
    e.stopPropagation();
    els.engineDropdown.classList.toggle("hidden");
  });

  document.addEventListener("click", () =>
    els.engineDropdown.classList.add("hidden")
  );

  els.engineDropdown.addEventListener("click", async (e) => {
    const opt = e.target.closest(".engine-option");
    if (!opt) return;
    currentEngine = opt.dataset.engine;
    els.engineIcon.src = engineIcons[currentEngine];
    els.engineDropdown.classList.add("hidden");
    await chrome.storage.sync.set({ searchEngine: currentEngine });
  });

  // Theme toggle
  els.btnTheme.addEventListener("click", () => {
    toggleTheme();
    updateThemeIcon();
  });

  // Initial theme icon update
  updateThemeIcon();

  // Settings Dialog
  els.btnSettings.addEventListener("click", async () => {
    const data = await chrome.storage.sync.get({
      wallpaperUrl: "",
      gridLayout: "8x5",
    });
    if (els.settingWallpaper)
      els.settingWallpaper.value = data.wallpaperUrl || "";

    // Update layout buttons state
    document.querySelectorAll(".layout-btn").forEach((btn) => {
      btn.classList.toggle("active", btn.dataset.value === data.gridLayout);
    });

    els.settingsDialog.showModal();
  });

  els.settingsClose.addEventListener("click", () => {
    els.settingsDialog.close();
  });

  // Layout Options Click Handler
  const layoutOptions = document.getElementById("layoutOptions");
  if (layoutOptions) {
    layoutOptions.addEventListener("click", async (e) => {
      const btn = e.target.closest(".layout-btn");
      if (!btn) return;

      // Update UI
      document
        .querySelectorAll(".layout-btn")
        .forEach((b) => b.classList.remove("active"));
      btn.classList.add("active");

      // Save and Apply immediately
      const newLayout = btn.dataset.value;
      await chrome.storage.sync.set({ gridLayout: newLayout });
      applyGridLayout(newLayout);
      renderPinnedGrid();
    });
  }

  // Clear All Button
  const btnClearAll = document.getElementById("btnClearAll");
  if (btnClearAll) {
    btnClearAll.addEventListener("click", async () => {
      if (confirm("确定要清空所有网格图标吗？此操作无法撤销。")) {
        pinnedSites = [];
        await savePinnedSites();
        renderPinnedGrid();
        updateSidebarAddButtons();
        els.settingsDialog.close();
      }
    });
  }

  // Removed btnSaveSettings listener as it is no longer needed
  /*
  els.btnSaveSettings.addEventListener("click", async () => {
    // ...
  });
  */

  // Edit Tile Dialog
  els.editTileClose.addEventListener("click", () => {
    saveAndCloseEditDialog();
  });

  // URL input - auto-fill name
  els.editTileUrl.addEventListener("change", async () => {
    const url = els.editTileUrl.value.trim();
    if (!url) return;

    // Auto-fill name if empty
    if (!els.editTileName.value) {
      // Show loading state
      els.editTileName.placeholder = "正在获取标题...";

      let title = null;
      try {
        // Ensure URL has protocol for fetch
        let fetchUrl = url;
        if (!/^https?:\/\//i.test(fetchUrl)) {
          fetchUrl = "https://" + fetchUrl;
        }

        const response = await fetch(fetchUrl);
        if (response.ok) {
          const text = await response.text();
          const doc = new DOMParser().parseFromString(text, "text/html");
          if (doc.title) {
            title = doc.title;
          }
        }
      } catch (e) {
        console.error("Failed to fetch title", e);
      }

      if (title) {
        els.editTileName.value = title;
      } else {
        // Fallback to domain
        let domain = getDomainFromUrl(url);
        domain = domain.charAt(0).toUpperCase() + domain.slice(1);
        els.editTileName.value = domain;
      }
      els.editTileName.placeholder = "网站名称";
    }

    // Update preview
    const idx = parseInt(els.editTileIndex.value);
    const bm =
      idx === -1
        ? { url: url, title: els.editTileName.value }
        : pinnedSites[idx];
    if (idx === -1) {
      bm.url = url;
      bm.title = els.editTileName.value;
    }
    updateEditTilePreview(bm);
    fetchAllIconSources(bm);
  });

  els.editTileIconUrl.addEventListener("input", () => {
    const idx = parseInt(els.editTileIndex.value);
    const bm = pinnedSites[idx];
    updateEditTilePreview(bm);
  });

  // Name input - update preview when typing
  els.editTileName.addEventListener("input", () => {
    const idx = parseInt(els.editTileIndex.value);
    const bm = pinnedSites[idx];
    updateEditTilePreview(bm);
  });

  // Color picker
  els.colorPicker.addEventListener("click", (e) => {
    const colorOpt = e.target.closest(".color-option");
    if (!colorOpt) return;

    const color = colorOpt.dataset.color;
    els.editTileBgColor.value = color;
    updateColorPickerSelection(color);

    // Update preview - force show letter icon
    const idx = parseInt(els.editTileIndex.value);
    const bm = pinnedSites[idx];
    showPreviewLetter(bm, color);
  });

  // File upload for custom icon
  els.editTileIconFile.addEventListener("change", async (e) => {
    const file = e.target.files[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = () => {
      const dataUrl = reader.result; // Base64 data URL
      els.editTileIconUrl.value = dataUrl;

      // Update icon sources and select upload
      currentEditIconSources.upload = dataUrl;
      const idx = parseInt(els.editTileIndex.value);
      selectIconSource("upload", pinnedSites[idx]);
    };
    reader.readAsDataURL(file);
  });

  // Close dialogs on backdrop click
  els.settingsDialog.addEventListener("click", (e) => {
    if (e.target === els.settingsDialog) els.settingsDialog.close();
  });
  els.editTileDialog.addEventListener("click", (e) => {
    if (e.target === els.editTileDialog) saveAndCloseEditDialog();
  });
}
