// Background service worker (MV3)
// Handles cross-origin requests for favicon fetching

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg?.type === "PING") {
    sendResponse({ ok: true, ts: Date.now() });
    return true;
  }

  // Fetch favicon from website HTML
  if (msg?.type === "FETCH_FAVICON") {
    fetchFaviconFromSite(msg.url)
      .then((iconUrl) => sendResponse({ ok: true, iconUrl }))
      .catch((err) => sendResponse({ ok: false, error: err.message }));
    return true; // Keep channel open for async response
  }
});

// Fetch and parse HTML to find favicon link tags
async function fetchFaviconFromSite(url) {
  try {
    const siteUrl = new URL(url);
    const baseUrl = siteUrl.origin;

    // Fetch the HTML page
    const response = await fetch(url, {
      method: "GET",
      headers: {
        "User-Agent":
          "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36",
      },
      signal: AbortSignal.timeout(5000),
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    const html = await response.text();

    // Parse HTML to find icon links
    const iconUrl = extractFaviconFromHtml(html, baseUrl);
    return iconUrl;
  } catch (err) {
    console.error("Failed to fetch favicon:", err);
    return null;
  }
}

// Extract favicon URL from HTML
function extractFaviconFromHtml(html, baseUrl) {
  // Match various link rel patterns for icons
  const patterns = [
    // Apple touch icon (usually high quality)
    /<link[^>]*rel=["']apple-touch-icon["'][^>]*href=["']([^"']+)["']/i,
    /<link[^>]*href=["']([^"']+)["'][^>]*rel=["']apple-touch-icon["']/i,
    // Icon with sizes (prefer larger)
    /<link[^>]*rel=["']icon["'][^>]*sizes=["']\d+x\d+["'][^>]*href=["']([^"']+)["']/i,
    /<link[^>]*href=["']([^"']+)["'][^>]*rel=["']icon["'][^>]*sizes=/i,
    // Shortcut icon
    /<link[^>]*rel=["']shortcut icon["'][^>]*href=["']([^"']+)["']/i,
    /<link[^>]*href=["']([^"']+)["'][^>]*rel=["']shortcut icon["']/i,
    // Standard icon
    /<link[^>]*rel=["']icon["'][^>]*href=["']([^"']+)["']/i,
    /<link[^>]*href=["']([^"']+)["'][^>]*rel=["']icon["']/i,
  ];

  for (const pattern of patterns) {
    const match = html.match(pattern);
    if (match && match[1]) {
      let iconUrl = match[1];
      // Handle relative URLs
      if (iconUrl.startsWith("//")) {
        iconUrl = "https:" + iconUrl;
      } else if (iconUrl.startsWith("/")) {
        iconUrl = baseUrl + iconUrl;
      } else if (!iconUrl.startsWith("http")) {
        iconUrl = baseUrl + "/" + iconUrl;
      }
      return iconUrl;
    }
  }

  return null;
}
