// AI classification helper.
// Strategy:
// - If user configured OpenAI API key in options, call it to categorize.
// - Otherwise, use a lightweight local heuristic as fallback.

export async function classifyBookmarks(bookmarks) {
  const settings = await chrome.storage.sync.get({
    provider: "openai",
    openaiKey: "",
    model: "gpt-4o-mini",
  });
  const hasKey = settings.provider === "openai" && settings.openaiKey;
  try {
    if (hasKey) {
      return await classifyWithOpenAI(bookmarks, settings);
    }
  } catch (e) {
    console.warn("OpenAI failed, fallback to heuristic", e);
  }
  return classifyHeuristic(bookmarks);
}

function classifyHeuristic(bookmarks) {
  // Group by primary domain as crude categories
  const groups = new Map();
  for (const bm of bookmarks) {
    let cat = "其他";
    try {
      const u = new URL(bm.url);
      const host = u.hostname.replace(/^www\./, "");
      const tld = host.split(".").slice(-2).join(".");
      cat = host.includes("github")
        ? "开发/GitHub"
        : host.includes("google")
        ? "搜索/工具"
        : host.includes("youtube")
        ? "视频/音频"
        : host.includes("bilibili")
        ? "视频/Bilibili"
        : host.includes("zhihu")
        ? "知识/社区"
        : host.includes("reddit")
        ? "社区/Reddit"
        : tld;
    } catch {}
    const list = groups.get(cat) || [];
    list.push({ category: cat, bookmark: bm });
    groups.set(cat, list);
  }
  return [...groups.values()].flat();
}

async function classifyWithOpenAI(bookmarks, settings) {
  const sys =
    `你是一个浏览器收藏夹整理助手。\n` +
    `请根据 URL 与标题，将书签划分到简洁的类别，例如：开发、设计、AI、新闻、视频、社区、效率、学习、工作、生活、购物、金融等。` +
    `请只输出 JSON 数组，每个元素形如 {"title":"...","url":"...","category":"..."}。类别数量保持适中（8-16）。`;
  const user = JSON.stringify(
    bookmarks.map((b) => ({ title: b.title, url: b.url })),
    null,
    0
  );
  const body = {
    model: settings.model || "gpt-4o-mini",
    messages: [
      { role: "system", content: sys },
      { role: "user", content: user },
    ],
    temperature: 0.2,
    response_format: { type: "json_object" },
  };
  const res = await fetch("https://api.openai.com/v1/chat/completions", {
    method: "POST",
    headers: {
      Authorization: `Bearer ${settings.openaiKey}`,
      "Content-Type": "application/json",
    },
    body: JSON.stringify(body),
  });
  if (!res.ok) throw new Error(`OpenAI HTTP ${res.status}`);
  const data = await res.json();
  const text = data.choices?.[0]?.message?.content || "{}";
  let parsed;
  try {
    parsed = JSON.parse(text);
  } catch {
    // Some providers wrap array; try to find an array substring
    const start = text.indexOf("[");
    const end = text.lastIndexOf("]");
    parsed = JSON.parse(text.slice(start, end + 1));
  }
  const arr = Array.isArray(parsed)
    ? parsed
    : parsed.items || parsed.data || [];
  return arr.map((x, i) => ({
    category: x.category || "其他",
    bookmark: bookmarks[i] || { title: x.title, url: x.url },
  }));
}
