# 浏览器扩展 - 图标获取策略文档（最终版）

## 核心架构：5 层优先级 + IndexedDB 缓存

图标获取流程**总是**遵循以下优先级，确保侧边栏、网格、编辑对话框的**完全一致**：

```
优先级 0️⃣  用户自定义图标   ⭐⭐⭐⭐⭐
         ↓ (无)
优先级 1️⃣  字母占位符        ⭐⭐⭐⭐ (即时显示)
         ↓ (后台异步)
优先级 2️⃣  IndexedDB 缓存    ⭐⭐⭐⭐⭐ (7 天 TTL)
         ↓ (无缓存)
优先级 3️⃣  Clearbit Logo API ⭐⭐⭐⭐⭐ (主力方案)
         ↓ (失败)
优先级 4️⃣  HTML 自动解析    ⭐⭐⭐⭐ (小众网站)
         ↓ (全失败)
优先级 5️⃣  保留字母占位符    ⭐⭐⭐⭐ (最终兜底)
```

---

## 优先级 0：用户自定义图标 ⭐⭐⭐⭐⭐

- **来源**: 用户上传/输入的 URL 或 base64 dataURL
- **字段**: `bm.customIcon` 或编辑对话的 `editTileIconUrl.value`
- **速度**: 极快（本地）
- **成功率**: 100%（信任用户输入）
- **特点**: 绕过所有异步流程，立即显示

**代码位置**：

```javascript
async function loadAndApplyIcon(container, { url, customIcon } = {}) {
  if (customIcon && customIcon !== "LETTER") {
    applyBackgroundImage(container, customIcon);
    return;
  }
  // ...
}
```

---

## 优先级 1：字母占位符 ⭐⭐⭐⭐

- **来源**: 书签标题的前 2 个字符（支持中文、emoji）
- **触发**: 页面加载时**立即**显示（同步）
- **背景色**: 根据域名 hash 生成一致的彩色背景
- **特点**: 永不空白、快速识别、离线可用

**代码位置**：

```javascript
function getTwoChars(text) {
  // 提取中文、emoji、数字等第一和第二个字符
  const matches = text.match(/[\p{L}\p{N}\p{Emoji}]/gu);
  // ...
}

function getIconColor(url) {
  // 根据域名 hash 保证同一网站始终同一颜色
  let hash = 0;
  for (let i = 0; i < url.length; i++) {
    hash = url.charCodeAt(i) + ((hash << 5) - hash);
  }
  return ICON_COLORS[Math.abs(hash) % ICON_COLORS.length];
}

function showLetterIcon(container, bm) {
  container.innerHTML = "";
  const letter = document.createElement("div");
  letter.className = "tile-letter";
  letter.textContent = getTwoChars(bm.title || new URL(bm.url).hostname);
  letter.style.background = bm.iconBgColor || getIconColor(bm.url);
  container.appendChild(letter);
}
```

---

## 优先级 2：IndexedDB 缓存 ⭐⭐⭐⭐⭐

**新增！** 使用 `indexedDB` 替代 `chrome.storage.local`，更适合存储二进制/大容量图标。

### 缓存结构

| 参数        | 说明                                     | 示例                                   |
| ----------- | ---------------------------------------- | -------------------------------------- |
| **Key**     | 域名 (domain)                            | `github.com`                           |
| **dataUrl** | Canvas 生成的 128px PNG 数据 URL（可选） | `data:image/png;base64,...`            |
| **url**     | 远程图标 URL（备用）                     | `https://logo.clearbit.com/github.com` |
| **source**  | 图标来源标记                             | `"clearbit"` \| `"html"`               |
| **ts**      | 缓存时间戳（毫秒）                       | `1701619200000`                        |

### 缓存特性

- **数据库**: `favicon_cache_db`
- **对象存储**: `icons`
- **TTL**: 7 天 (`7 * 24 * 60 * 60 * 1000` ms)
- **过期策略**: 懒删除 (访问时检查 TTL，过期则删除)

### 缓存操作代码

```javascript
const ICON_CACHE_DB_NAME = "favicon_cache_db";
const ICON_CACHE_STORE_NAME = "icons";
const ICON_CACHE_TTL_MS = 7 * 24 * 60 * 60 * 1000; // 7 days

function openIconCacheDB() {
  return new Promise((resolve, reject) => {
    const req = indexedDB.open(ICON_CACHE_DB_NAME, 1);
    req.onupgradeneeded = (e) => {
      const db = e.target.result;
      if (!db.objectStoreNames.contains(ICON_CACHE_STORE_NAME)) {
        db.createObjectStore(ICON_CACHE_STORE_NAME);
      }
    };
    req.onsuccess = () => resolve(req.result);
    req.onerror = () => reject(req.error);
  });
}

async function getIconCacheEntry(domain) {
  // 检查 indexedDB，若过期则返回 null 并在后台删除
}

async function setIconCacheEntry(domain, entry) {
  // 存储到 indexedDB，自动添加时间戳
}
```

**优势**：

- ✅ 可存储二进制 blob 数据（远大于 `chrome.storage.local` 的 10KB 限制）
- ✅ 异步 API，不阻塞主线程
- ✅ 自动 TTL 管理，懒删除避免垃圾

---

## 优先级 3：Clearbit Logo API ⭐⭐⭐⭐⭐

**主力方案** — 覆盖主流网站，高质量，无需认证。

### 端点

```
https://logo.clearbit.com/{domain}?size=256
```

### 特性

| 特性       | 说明                                         |
| ---------- | -------------------------------------------- |
| **质量**   | 512x512 或更高 PNG                           |
| **覆盖率** | 主流网站 > 95%                               |
| **错误**   | 无 404 — 总是返回 200 (未找到时返回通用图标) |
| **速度**   | 中等（CDN）                                  |
| **缩放**   | `?size=256` 获取最佳尺寸                     |

### 尝试流程

```javascript
async function loadAndApplyIcon(container, { url, customIcon } = {}) {
  // ... 检查缓存后

  (async () => {
    try {
      const cb = faviconClearbit(url);
      if (cb && (await testImageUrl(cb))) {
        // 尝试用 Canvas 归一化为 128px dataURL（更好的网络传输体验）
        const data = await resizeImageToDataUrl(cb, 128);
        if (data) {
          await setIconCacheEntry(domain, {
            dataUrl: data, // ← 存储 dataURL
            source: "clearbit",
          });
          applyBackgroundImage(container, data);
          return;
        }
        // 若 Canvas 失败（CORS），直接使用远程 URL
        await setIconCacheEntry(domain, { url: cb, source: "clearbit" });
        applyBackgroundImage(container, cb);
        return;
      }
    } catch (err) {
      console.warn("Clearbit attempt failed", err);
    }
    // ...
  })();
}
```

---

## 优先级 4：网站 HTML 自动解析 ⭐⭐⭐⭐

**通用兜底** — 从网站源码获取官方图标，适配小众网站。

### 解析规则

`background.js` 的 `FETCH_FAVICON` 消息处理器会依次扫描：

```html
<link rel="icon" ... />
<link rel="shortcut icon" ... />
<link rel="apple-touch-icon" ... />
<link rel="apple-touch-icon-precomposed" ... />
<link rel="mask-icon" ... />
<meta property="og:image" ... />
<meta name="twitter:image" ... />
```

### 优化策略

- 按尺寸排序，优先 128px ~ 256px
- `.svg` 优先（矢量，可完美缩放）
- 其次 `.png`，再 `.ico`

### 前端调用

```javascript
async function fetchFaviconFromSite(url) {
  try {
    const response = await chrome.runtime.sendMessage({
      type: "FETCH_FAVICON",
      url: url,
    });
    if (response?.ok && response.iconUrl) return response.iconUrl;
    return null;
  } catch (err) {
    console.error("Failed to fetch favicon from site:", err);
    return null;
  }
}
```

**限制**：

- ⚠️ 需要跨域请求 → background service worker 隔离
- ⚠️ CORS 限制可能导致解析失败
- ⚠️ 响应较慢（完整 HTML 解析）

---

## 优先级 5：保留字母占位符 ⭐⭐⭐⭐

若所有来源都失败，图标保留为字母占位符。**永不空白**。

```javascript
// 若优先级 3、4 都失败，则结束异步尝试
// container 仍显示字母占位符（已由调用方提前设置）
```

---

## 图像归一化：Canvas 缩放到 128px

为统一展示效果并减小网络体积，**尝试**将远程图标通过 Canvas 缩放为 128x128px 的 dataURL：

```javascript
async function resizeImageToDataUrl(imageUrl, size = 128) {
  try {
    const resp = await fetch(imageUrl, { mode: "cors" });
    const blob = await resp.blob();
    const img = await new Promise((resolve, reject) => {
      const url = URL.createObjectURL(blob);
      const i = new Image();
      i.crossOrigin = "anonymous";
      i.onload = () => {
        URL.revokeObjectURL(url);
        resolve(i);
      };
      i.onerror = (e) => {
        URL.revokeObjectURL(url);
        reject(e);
      };
      i.src = url;
    });

    const canvas = document.createElement("canvas");
    canvas.width = size;
    canvas.height = size;
    const ctx = canvas.getContext("2d");
    ctx.clearRect(0, 0, size, size);

    // 保持宽高比，居中绘制
    const ratio = Math.min(size / img.width, size / img.height);
    const w = img.width * ratio;
    const h = img.height * ratio;
    const x = (size - w) / 2;
    const y = (size - h) / 2;
    ctx.drawImage(img, x, y, w, h);
    return canvas.toDataURL("image/png");
  } catch (err) {
    // CORS/tainted canvas 失败 → 回退到原始 URL
    return null;
  }
}
```

**效果**：

- ✅ 图像质量统一（128px）
- ✅ 传输大小更小（dataURL 对小图可接受）
- ✅ CORS 限制时自动回退到原始 URL

---

## UI 集成：三个调用点完全一致

### 1. 网格 (`createTile`)

```javascript
function createTile(bm, idx) {
  const icon = document.createElement("div");
  icon.className = "tile-icon";

  // 立即显示字母占位符
  showLetterIcon(icon, bm);

  // 后台异步尝试替换为真实图标
  loadFaviconWithFallback(icon, bm).catch(() => {
    // 若全失败，保留字母占位符（已显示）
  });

  // ...
}
```

### 2. 侧边栏 (`renderFolderOrItem`)

```javascript
const iconWrap = document.createElement("div");
iconWrap.className = "sb-item-icon";

// 立即显示字母占位符
iconWrap.textContent = getTwoChars(node.title);
iconWrap.style.background = hashColor(node.url);

// 后台异步尝试替换
loadSidebarFavicon(iconWrap, node);
```

### 3. 编辑对话预览 (`updateEditTilePreview`)

```javascript
function updateEditTilePreview(bm) {
  const iconUrl = els.editTileIconUrl.value || bm.customIcon;

  if (iconUrl === "LETTER") {
    showPreviewLetter(bm, bgColor);
    return;
  }

  if (iconUrl) {
    // 用户提供的 URL，尝试直接显示
    const img = document.createElement("img");
    img.src = iconUrl;
    img.onerror = () => showPreviewLetter(bm, bgColor);
    els.editTileIcon.appendChild(img);
  } else {
    // 立即显示字母占位符
    showPreviewLetter(bm, bgColor);
    // 后台异步尝试替换
    loadPreviewFavicon(bm, bgColor);
  }
}
```

---

## 已弃用方式（为什么不用）

| 方式                  | 原因        | 说明                              |
| --------------------- | ----------- | --------------------------------- |
| **DuckDuckGo**        | 404 率高    | 中文/小众网站返回 404，污染控制台 |
| **直接 /favicon.ico** | 失败率 50%+ | 很多网站不提供或已弃用            |
| **Google S2**         | 地区限制    | 部分地区 GFW 阻止，加载不稳定     |

---

## 性能指标对比

| 方式          | 加载速度 | 质量      | 成功率     | 错误   | 建议   |
| ------------- | -------- | --------- | ---------- | ------ | ------ |
| 自定义        | 极快     | 用户控制  | 100%       | 无     | 优先   |
| 字母占位      | 极快     | 中        | 100%       | 无     | 必需   |
| **缓存**      | **极快** | **保存**  | **100%**   | **无** | ✅     |
| **Clearbit**  | **中等** | **高**    | **60-70%** | **无** | ✅     |
| **HTML 解析** | **慢**   | **中-高** | **50-60%** | **无** | ✅     |
| DuckDuckGo\*  | 中等     | 低        | 30-40%     | 404 ❌ | ✗ 弃用 |

---

## 调试方法

### 查看当前缓存内容

```javascript
// 在扩展控制台运行
const db = await new Promise((resolve, reject) => {
  const req = indexedDB.open("favicon_cache_db", 1);
  req.onsuccess = () => resolve(req.result);
  req.onerror = () => reject(req.error);
});
const tx = db.transaction("icons", "readonly");
const store = tx.objectStore("icons");
const allReq = store.getAll();
allReq.onsuccess = () => console.log("缓存内容：", allReq.result);
```

### 测试 Clearbit 可用性

```javascript
fetch("https://logo.clearbit.com/github.com?size=256", { mode: "no-cors" })
  .then(() => console.log("✓ Clearbit 可用"))
  .catch(() => console.log("✗ Clearbit 不可用"));
```

### 测试 HTML 解析

```javascript
// 在 background script 控制台运行
chrome.runtime.onMessage.addListener((msg, sender, reply) => {
  if (msg.type === "FETCH_FAVICON") {
    console.log("Received FETCH_FAVICON for:", msg.url);
    reply({ ok: true, iconUrl: "..." });
  }
});
```

### 清空缓存

```javascript
// 在扩展控制台运行
const req = indexedDB.deleteDatabase("favicon_cache_db");
req.onsuccess = () => console.log("✓ 缓存已清空");
```

---

## 总结：最优的三层防御

```
第 1 层：字母占位符（即时，稳定）
        ↓
第 2 层：Clearbit + HTML 解析（后台，优化）
        + IndexedDB 缓存（加速重复访问）
        ↓
第 3 层：字母占位符回退（始终有备选）
```

**核心收获**：

- ✅ 用户永不看到空白图标
- ✅ 异步加载，不阻塞 UI
- ✅ 无控制台 404 错误
- ✅ 7 天智能缓存，加速重复访问
- ✅ 网格、侧边栏、编辑对话**完全一致**
